/* eslint-disable default-case,complexity */
define(['underscore', 'backbone', 'moment'], function(_, Backbone, moment) {
    'use strict';

    // model
    var bookedAppointment = Backbone.Model.extend({
        isCancelled: function() {
            return hasCancelledStatus(this.get('currentStatus'));
        },
        parse: function(response, options) {
            /**
             * Consider rewriting such that instead of ternaries we have a case switch here and a parse
             * that says if the model has one of these unique properties set type to VAR/CC/Future implementations
             *
             * that way instead of ternaries
             - we have a switch that supports more than two types of appointment information.
             */

            response.siteCode = options.siteCode || '';
            // Placeholder until AC is more clear
            response.facilityName = response.clinic ? response.clinic.friendlyName : response.name.firstName;
            // Placeholder until AC is more clear
            response.clinic = response.clinic ? response.clinic : {};
            response.clinic.friendlyName = response.clinic ? response.clinic.friendlyName : response.name.firstName;
            return response;
        },
    });

    // Share this with the model above
    function hasCancelledStatus(status) {

        switch (status) {
        case 'NO-SHOW':
        case 'CANCELLED BY CLINIC':
        case 'NO-SHOW & AUTO RE-BOOK':
        case 'CANCELLED BY CLINIC & AUTO RE-BOOK':
        case 'INPATIENT APPOINTMENT':
        case 'CANCELLED BY PATIENT':
        case 'CANCELLED BY PATIENT & AUTO-REBOOK':
        case 'NO ACTION TAKEN':
            return true;
        }
        return false;
    }


    return Backbone.Collection.extend({
        // Share it here to have access to it for testing
        _hasCancelledStatus: hasCancelledStatus,

        model: bookedAppointment,

        comparator: function(model) {
            return new Date(model.get('appointmentTime'));
        },

        // This function will be easier to unit test than parse would be
        filterItem: function(appointment) {
            var isCancelled = hasCancelledStatus(appointment.currentStatus);
            var today;
            var appointmentTime;
            if (isCancelled) {
                return !isCancelled;
            }

            if (_.isUndefined(appointment.appointmentTime)) {
                return false;
            }


            if (appointment.appointmentTime) {
                today = new Date();
                appointmentTime = new Date(appointment.appointmentTime);
                this.fixTime(appointment);
                return appointmentTime > today;
            }

            return appointment;
        },
        fixTime: function(appointment) {
            var time;
            var hours;

            // Only do the extra work if it has the timeZone fix needed (CC)
            if (appointment.timeZone) {
                time = moment(appointment.appointmentTime);
                hours = appointment.timeZone.split(':')[0];
                time = time.add(hours, 'hours');

                appointment.appointmentTime = time.format('MM/DD/YYYY HH:mm:ss');
            }
        },

        // As far as i know response should always be defined when backbone invokes it.
        parse: function(response) {
            var bookedCollection = response.bookedAppointmentCollections || [];
            var appointments = _.map(bookedCollection, function(collection) {
                return _.filter(collection[collection.collectionName], this.filterItem.bind(this));
            }.bind(this));

            // _.flatten([[1, 2], [3, 4], [5,6,7], []])  => [1, 2, 3, 4, 5, 6, 7]
            return _.flatten(appointments);
        },
    });
});
